/**
 * Heslar Tooltips JavaScript
 * Enhanced tooltip functionality with dynamic content loading
 */
(function($) {
    'use strict';

    class HeslarTooltips {
        constructor() {
            this.init();
        }

        init() {
            this.bindEvents();
            this.createTooltipElements();
        }

        bindEvents() {
            $(document).on('mouseenter', '.heslar-tooltip', this.showTooltip.bind(this));
            $(document).on('mouseleave', '.heslar-tooltip', this.hideTooltip.bind(this));
            $(window).on('resize', this.repositionTooltips.bind(this));
        }

        createTooltipElements() {
            $('.heslar-tooltip').each(function() {
                const $tooltip = $(this);
                const tooltipText = $tooltip.attr('data-tooltip') || $tooltip.attr('title');
                
                if (tooltipText && !$tooltip.find('.heslar-tooltip-content').length) {
                    const $tooltipContent = $('<div class="heslar-tooltip-content"></div>');
                    $tooltipContent.html(tooltipText);
                    $tooltip.append($tooltipContent);
                    
                    // Remove title attribute to prevent default browser tooltip
                    $tooltip.removeAttr('title');
                }
            });
        }

        showTooltip(event) {
            const $tooltip = $(event.currentTarget);
            const $content = $tooltip.find('.heslar-tooltip-content');
            
            if ($content.length) {
                // Position tooltip based on viewport
                this.positionTooltip($tooltip, $content);
                
                // Show tooltip with animation
                $content.stop(true, true).fadeIn(200);
            }
        }

        hideTooltip(event) {
            const $tooltip = $(event.currentTarget);
            const $content = $tooltip.find('.heslar-tooltip-content');
            
            $content.stop(true, true).fadeOut(150);
        }

        positionTooltip($tooltip, $content) {
            const windowWidth = $(window).width();
            const tooltipOffset = $tooltip.offset();
            const tooltipWidth = $tooltip.outerWidth();
            const contentWidth = $content.outerWidth();
            
            // Reset positioning classes
            $content.removeClass('tooltip-left tooltip-right');
            
            // Check if tooltip would go off-screen
            const leftEdge = tooltipOffset.left - (contentWidth / 2) + (tooltipWidth / 2);
            const rightEdge = leftEdge + contentWidth;
            
            if (leftEdge < 10) {
                $content.addClass('tooltip-left');
            } else if (rightEdge > windowWidth - 10) {
                $content.addClass('tooltip-right');
            }
        }

        repositionTooltips() {
            $('.heslar-tooltip-content:visible').each((index, element) => {
                const $content = $(element);
                const $tooltip = $content.parent();
                this.positionTooltip($tooltip, $content);
            });
        }

        // Method to dynamically add tooltips to new content
        refresh() {
            this.createTooltipElements();
        }
    }

    // Initialize when document is ready
    $(document).ready(function() {
        window.heslarTooltips = new HeslarTooltips();
        
        // Refresh tooltips when new content is loaded (for AJAX content)
        $(document).on('content-updated', function() {
            window.heslarTooltips.refresh();
        });
    });

    // Accessibility improvements
    $(document).on('keydown', '.heslar-tooltip', function(e) {
        if (e.keyCode === 13 || e.keyCode === 32) { // Enter or Space
            e.preventDefault();
            $(this).trigger('mouseenter');
        }
        if (e.keyCode === 27) { // Escape
            $(this).trigger('mouseleave');
        }
    });

})(jQuery);